#ifndef __CAudioBuffer__
#define __CAudioBuffer__

//	===========================================================================

#include <Basics/CCountedObject.hpp>
#include <Exceptions/CException.hpp>
#include <MathTools/CMathTools.hpp>

//	===========================================================================

using Exponent::Basics::CCountedObject;
using Exponent::Exceptions::CException;
using Exponent::MathTools::CMathTools;

//	===========================================================================

namespace Exponent
{
	namespace Audio
	{
		/**
		 * @deprecated Soon to be removed.. i dont advise you use this class
		 * @see TAudioBuffer
		 * @class CAudioBuffer CAudioBuffer.hpp
		 * @brief A stereo audio buffer
		 *
		 * Wraps up a stereo double audio buffer\n
		 * NOTE : Class is aggressively inlined for improved performance
		 *
		 * @date 31/01/2005
		 * @author Paul Chana
		 * @version 1.0.0 Initial version
		 * @version 1.0.1 Added RMS functions
		 * @version 1.0.2 Added Default constructor
		 *
		 * @note All contents of this source code are copyright 2005 Exp Digital Uk.\n
		 * This source file is covered by the licence conditions of the Infinity API. You should have recieved a copy\n
		 * with the source code. If you didnt, please refer to http://www.expdigital.co.uk
		 * All content is the Intellectual property of Exp Digital Uk.\n
		 * Certain sections of this code may come from other sources. They are credited where applicable.\n
		 * If you have comments, suggestions or bug reports please visit http://support.expdigital.co.uk
		 *
		 * $Id: CAudioBuffer.hpp,v 1.7 2007/02/08 21:08:09 paul Exp $
		 */
		class CAudioBuffer : public CCountedObject
		{
			/** @cond */
			EXPONENT_CLASS_DECLARATION;
			/** @endcond */

//	===========================================================================

		public:

//	===========================================================================

			const static double CAUDIO_BUFFER_DENORMAL;		/**< Denormal value */

//	===========================================================================

			/**
			 * @enum EStereoChannels
			 * @brief Enumerations with defaults for stereo and mono information
			 */
			enum EStereoChannels
			{
				e_left = 0,		/**< Left channel */
				e_right,		/**< Right channel */

				e_mono = 1,		/**< Mono signal */
				e_stereo,		/**< Stereo signal */
			};

//	===========================================================================

			/**
			 * Construction
			 * @param bufferSize The size of the audio buffer to create in samples
			 */
			CAudioBuffer(const long bufferSize);

			/**
			 * Construction
			 */
			CAudioBuffer();

			/**
			 * Destruction
			 */
			virtual ~CAudioBuffer();

//	===========================================================================

			/**
			 * Get the raw wave information
			 * @retval double** The mutable audio data
			 */
			double **getMutableAudioData() const { return m_waveData; }

			/**
			 * Get a constant raw wave information
			 * @retval double** The audio data
			 */
			const double **getAudioData() const { return (const double **)&m_waveData[0]; }

			/**
			 * Get channel data
			 * @param channel The channel you want - NOT RANGE CHECKED!
			 * @retval double* The channel data
			 */
			FORCEINLINE double *getChannel(const long channel) { return m_waveData[channel]; }

			/**
			 * Get a specific sample - not recommended for high speed
			 * @param channel The channel to get the sample from
			 * @param index The sample index to get
			 * @retval double The sample
			 */
			FORCEINLINE double getSample(const long channel, const long index) { return m_waveData[channel][index]; }

//	===========================================================================

			/**
			 * Initialise the audio data (deletes old buffers)
			 * @param bufferSize The size of the audio buffer to use
			 */
			void initialise(const long bufferSize);

			/**
			 * Uninitialise the audio data
			 */
			void uninitialise();

			/**
			 * Uninit then init
			 * @param bufferSize The size of the audio buffer to use
			 */
			void reInitialise(const long bufferSize);

//	===========================================================================

			/**
			 * Zero the wave form
			 */
			void FORCEINLINE clearWaveForm()
			{
				const size_t size = m_bufferSize * sizeof(double);
				memset(m_waveData[0], 0, size);
				memset(m_waveData[1], 0, size);
			}

//	===========================================================================

			/**
			 * Merge left and right
			 */
			void FORCEINLINE mergeChannels()
			{
				for (long i = 0; i < m_bufferSize; i++)
				{
					m_waveData[0][i] = m_waveData[1][i] = ((m_waveData[0][i] * 0.5) + (m_waveData[1][i] * 0.5));
				}
			}

			/**
			 * Merge with another buffer
			 * @param buffer The buffer to merge with
			 * @exception CException thrown if the bufer to be merged with has a different sized buffer
			 */
			void FORCEINLINE mergeWithBuffer(CAudioBuffer *buffer)
			{
				if (buffer->getBufferSize() != m_bufferSize)
				{
					throw CException("Buffer sizes do not match", "CAudioBuffer::mergeWithBuffer(CAudioBuffer *)");
				}

				double *left   = m_waveData[0];
				double *right  = m_waveData[1];
				double *leftF  = buffer->getChannel(e_left);
				double *rightF = buffer->getChannel(e_right);

				for (long i = 0; i < m_bufferSize; i++)
				{
					left[i]  += leftF[i];
					right[i] += rightF[i];
				}
			}

//	===========================================================================

			/**
			 * Merge in to passed output
			 * @param output The buffer to output to
			 * @param numberOfSamples The number of samples to process
			 */
			void FORCEINLINE accumulateInToFloatData(float **output, const long numberOfSamples)
			{
				double *left  = m_waveData[0];
				double *right = m_waveData[1];
				float *leftF  = output[0];
				float *rightF = output[1];

				for (long i = 0; i < numberOfSamples; i++)
				{
					leftF[i]  += (float)left[i];
					rightF[i] += (float)right[i];
				}
			}

			/**
			 * Merge and store in the passed buffer
			 * @param output The output buffer
			 * @param numberOfSamples The number of samples to process
			 * @exception CException If the output buffer is NULL
			 */
			void FORCEINLINE mergeToBuffer(double *output, const long numberOfSamples)
			{
				if (output == NULL)
				{
					throw CException("Output buffer is NULL", "CAudioBuffer::mergeToBuffer(double *, const long)");
				}

				double *left  = m_waveData[0];
				double *right = m_waveData[1];

				for (long i = 0; i < numberOfSamples; i++)
				{
					output[i] = (left[i] * 0.5) + (right[i] * 0.5);
				}
			}

//	===========================================================================

			/**
			 * Get the blocksize
			 * @retval long The buffer size
			 */
			long getBufferSize() const { return m_bufferSize; }

//	===========================================================================

			/**
			 * Set from another audiobuffer
			 * @param buffer The buffer to copy
			 * @exception CException thrown if the bufer to be merged with has a different sized buffer
			 */
			void FORCEINLINE setFromAudioBuffer(CAudioBuffer *buffer)
			{
				if (buffer->getBufferSize() != m_bufferSize)
				{
					throw CException("Buffer sizes do not match", "CAudioBuffer::setFromAudioBuffer(CAudioBuffer *)");
				}

				memcpy(m_waveData[0], buffer->getChannel(e_left),  m_bufferSize * sizeof(double));
				memcpy(m_waveData[1], buffer->getChannel(e_right), m_bufferSize * sizeof(double));
			}

			/**
			 * Set the audio data from float data
			 * @param waveData The data to copy
			 * @param numberOfSamples The number of samples to process
			 */
			void FORCEINLINE setWaveDataFromFloatData(float **waveData, const long numberOfSamples)
			{
				double *left  = m_waveData[0];
				double *right = m_waveData[1];

				float *leftF  = waveData[0];
				float *rightF = waveData[1];

				for (long i = 0; i < numberOfSamples; i++)
				{
					left[i]  = (double)leftF[i];
					right[i] = (double)rightF[i];
				}
			}

			/**
			 * Set the audio data from float data and denormal
			 * @param waveData The data to copy
			 * @param numberOfSamples The number of samples to process
			 */
			void FORCEINLINE setWaveDataFromFloatDataAndDenormal(float **waveData, const long numberOfSamples)
			{
				double *left  = m_waveData[0];
				double *right = m_waveData[1];

				float *leftF  = waveData[0];
				float *rightF = waveData[1];

				for (long i = 0; i < numberOfSamples; i++)
				{
					left[i]  = (double)leftF[i]  + CAUDIO_BUFFER_DENORMAL;
					right[i] = (double)rightF[i] + CAUDIO_BUFFER_DENORMAL;
				}
			}

			/**
			 * Copy to a float wave
			 * @param waveData The buffer to copy to
			 * @param numberOfSamples The number of samples to copy
			 */
			void FORCEINLINE copyToFloatBufferFromWaveData(float **waveData, const long numberOfSamples)
			{
				double *left  = m_waveData[0];
				double *right = m_waveData[1];

				float *leftF  = waveData[0];
				float *rightF = waveData[1];

				for (long i = 0; i < numberOfSamples; i++)
				{
					leftF[i]  = (float)left[i];
					rightF[i] = (float)right[i];
				}
			}

//	===========================================================================

			/**
			 * Denormal this buffer of audio
			 * @param numberOfSamples How many samples to data should be denormaled?
			 */
			void FORCEINLINE denormal(const long numberOfSamples)
			{
				double *left  = m_waveData[0];
				double *right = m_waveData[1];
				for (long i = 0; i < numberOfSamples; i++)
				{
					left[i]  += CAUDIO_BUFFER_DENORMAL;
					right[i] += CAUDIO_BUFFER_DENORMAL;
				}
			}

			/**
			 * Change the gain
			 * @param volume The volume
			 */
			void FORCEINLINE changeGain(const double volume)
			{
				double *left  = m_waveData[0];
				double *right = m_waveData[1];
				for (long i = 0; i < m_bufferSize; i++)
				{
					left[i]  *= volume;
					right[i] *= volume;
				}
			}

			/**
			 * Get the maximum samples for each channel
			 * @param left filled with the left channels samples
			 * @param right filled with the right channels samples
			 */
			void getMaxSamples(double &left, double &right);

			/**
			 * Apply a tanh limiter to the signal. Applied to whole buffer
			 * @see CMathTools
			 */
			void FORCEINLINE applyTanhLimiting()
			{
				double *left  = m_waveData[0];
				double *right = m_waveData[1];
				for (long i = 0; i < m_bufferSize; i++)
				{
					left[i]  = CMathTools::tanhApproximation(left[i]);
					right[i] = CMathTools::tanhApproximation(right[i]);
				}
			}

			/**
			 * Compute the RMS envelope value
			 * @param buffer The buffer to compute it on (expects size 32)
			 * @param numberOfSamplesToProcess <= the size of buffer
			 * @retval double The RMS value
			 */
			FORCEINLINE static double computeRMS(const double *buffer, const long numberOfSamplesToProcess)
			{
				// Compute the squares
				double mean1 = 0.0, mean2 = 0.0, mean3 = 0.0, mean4 = 0.0;
				for (long i = 0; i < numberOfSamplesToProcess - 4; i+=4)
				{
					mean1 += buffer[i]     * buffer[i];
					mean2 += buffer[i + 1] * buffer[i + 1];
					mean3 += buffer[i + 2] * buffer[i + 2];
					mean4 += buffer[i + 3] * buffer[i + 3];
				}
				/*for (long i = 0; i < numberOfSamplesToProcess; i++)
				{
					mean += buffer[i] * buffer[i];
				}*/
				return sqrt((mean1 + mean2 + mean3 + mean4) / numberOfSamplesToProcess);

				// Compute the mean
				//mean /= numberOfSamplesToProcess;

				// Return the RMS
				//return sqrt(mean);
			}

			/**
			 * Compute the RMS envelope value
			 * @param buffer The buffer to compute it on (expects size 32)
			 * @param numberOfSamplesToProcess <= the size of buffer
			 * @retval double The RMS value
			 */
			FORCEINLINE static double computeRMS(const float *buffer, const long numberOfSamplesToProcess)
			{
				// Compute the squares
				double mean = 0.0;
				for (long i = 0; i < numberOfSamplesToProcess; i++)
				{
					mean += (double)(buffer[i] * buffer[i]);
				}

				// Compute the mean
				mean /= numberOfSamplesToProcess;

				// Return the RMS
				return sqrt(mean);
			}

//	===========================================================================

		protected:

//	===========================================================================

			double **m_waveData;		/**< The wave data */
			long m_bufferSize;			/**< The buffer size */
		};
	}
}

#endif	// End of CAudioBuffer.hpp